/* mmc.c - mmap cache
**
** Copyright  1998,2001 by Jef Poskanzer <jef@acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <syslog.h>
#include <sys/queue.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <limits.h>

#ifdef HAVE_MMAP
#include <sys/mman.h>
#endif /* HAVE_MMAP */

#include "mmc.h"
#include "libhttpd.h"

#ifndef __FreeBSD__
#undef USE_SENDFILE
#endif

#if defined(HAVE_KQUEUE) && defined(USE_SENDFILE)
#define USE_KQUEUE_VNODE 1
#include "fdwatch.h"
#endif


/* Defines. */
#ifndef DEFAULT_EXPIRE_AGE
#define DEFAULT_EXPIRE_AGE 600
#endif
#ifndef DESIRED_FREE_COUNT
#define DESIRED_FREE_COUNT 100
#endif
#ifndef DESIRED_MAX_MAPPED_FILES
#define DESIRED_MAX_MAPPED_FILES 2000
#endif
#ifndef INITIAL_HASH_SIZE
#define INITIAL_HASH_SIZE (1 << 10)
#endif

#define max(a,b) ((a)>(b)?(a):(b))
#define min(a,b) ((a)<(b)?(a):(b))


/* The Map struct. */
typedef struct MapStruct {
    ino_t ino;
    dev_t dev;
    off_t size;
    time_t ctime;
    int refcount;
    time_t reftime;
#ifdef USE_SENDFILE
    int fd;
#ifdef USE_KQUEUE_VNODE
    char *filename; /* Use this to implement a stat cache */
    uint32_t fn_hash; /* Save the full 32-bit hash for faster compares */
    struct stat sb;
    int sbvalidity; /* this is set to 0 when first entered
		     * 1 after first confirmed (since there is a race
		     * between when the statbuf is entered and when it is
		     * watched), and -1 when invalidated */
    TAILQ_ENTRY(MapStruct) statnext; /* hash table entry */
#endif
#endif
    void* addr;
    unsigned int hash;
    int hash_idx;
    char mtime[40]; /* RFC1123 fmt of the Last-Modified time */

  /* NOTE: Do not cache fileInfo if encodings are in use! */
    char *fileType; /* hold hc->type; not malloc'ed */
    char fileInfo[500]; /* hold type, lmt, len, etc. here */
    int  fileInfoLen;/* length of fileInfo */
  
    struct MapStruct* next;
    TAILQ_ENTRY(MapStruct) lruList; /* LRU list of entries with refcount 0 */
    } Map;

/* Globals. */
static Map* maps = (Map*) 0;
static Map* free_maps = (Map*) 0;
static int alloc_count = 0, map_count = 0, lru_count = 0, free_count = 0;
static Map** hash_table = (Map**) 0;
static int hash_size;
static unsigned int hash_mask;
static time_t expire_age = DEFAULT_EXPIRE_AGE;
int mmc_maxmaps = 40000; /* can be set with -m; should default to some
			  * fraction of the sysctl value. Oh well. */

/* LRU list of entries with refcount 0 */
static TAILQ_HEAD(, MapStruct) mapLRU =  TAILQ_HEAD_INITIALIZER(mapLRU);

/* Forwards. */
static void really_unmap( Map** mm );
static int check_hash_size( void );
static int add_hash( Map* m );
static Map* find_hash( ino_t ino, dev_t dev, off_t size, time_t ctime );
static unsigned int hash( ino_t ino, dev_t dev, off_t size, time_t ctime );

#ifdef USE_KQUEUE_VNODE
/* stat hashing */
static uint32_t mmc_stathash( const char *filename );
static void mmc_statadd( Map *map );
static void mmc_statdel( Map *map );
#endif

void*
mmc_map( Map **mapp, char* filename, httpd_conn *hc, struct timeval* nowP, char **mtime, uint32_t *fnhashP )
    {
    time_t now;
    struct stat sb;
    Map* m;
    int fd;

#ifdef DEBUG_MMC
    fprintf(stderr, __FUNCTION__ ":%d map_count %d lru_count %d\n",
	    __LINE__, map_count, lru_count);
#endif
    
    /* Stat the file, if necessary. */
    if ( hc != NULL )
	sb = hc->sb;
    else
	{
	if ( stat( filename, &sb ) != 0 )
	    {
	    syslog( LOG_ERR, "stat - %m" );
	    return (void*) 0;
	    }
	}

    /* Get the current time, if necessary. */
    if ( nowP != (struct timeval*) 0 )
	now = nowP->tv_sec;
    else
	now = time( (time_t*) 0 );

    /* See if we have it mapped already, via the hash table. */
    if ( check_hash_size() < 0 )
	{
	syslog( LOG_ERR, "check_hash_size() failure" );
	return (void*) 0;
	}
    m = find_hash( sb.st_ino, sb.st_dev, sb.st_size, sb.st_ctime );
    if ( m != (Map*) 0 )
	{
	/* Yep.  Just return the existing map */
	  
	if (0 == m->refcount) {
	  /* remove it from the LRU list */
	  TAILQ_REMOVE(&mapLRU, m, lruList);
	  lru_count--;
	}
	++m->refcount;
        /* hykim: mmc opt1 */
        *mapp = m;

	m->reftime = now;
	*mtime = m->mtime;

	hc->type = m->fileType;
	hc->fileInfo = m->fileInfo;
	hc->fileInfoLen = m->fileInfoLen;
	
#ifdef DEBUG_MMC
    fprintf(stderr, __FUNCTION__ ":%d map_count %d lru_count %d\n",
	    __LINE__, map_count, lru_count);
#endif
    
#ifdef USE_SENDFILE
	return (&m->fd);
#else
	return m->addr;
#endif
	}

    /* We are going to map a new file, so first consider unmapping an old
     * one if we don't have any more available to us. */
    if ((map_count >= mmc_maxmaps)  && (lru_count > 0)) {
      Map *last;

      last = TAILQ_FIRST(&mapLRU);
      TAILQ_REMOVE(&mapLRU, last, lruList);
      lru_count--;
      really_unmap(&last);
      
#ifdef DEBUG_MMC
      fprintf(stderr, __FUNCTION__ ":%d map_count %d lru_count %d\n",
	      __LINE__, map_count, lru_count);
#endif    
    }
    
    /* Open the file. */    
    fd = open( filename, O_RDONLY );
    if ( fd < 0 )
	{
	syslog( LOG_ERR, "open - %m" );
	return (void*) 0;
	}

    /* Find a free Map entry or make a new one. */
    if ( free_maps != (Map*) 0 )
	{
	m = free_maps;
	free_maps = m->next;
	--free_count;
	}
    else
	{
	m = (Map*) malloc( sizeof(Map) );
	if ( m == (Map*) 0 )
	    {
	    (void) close( fd );
	    syslog( LOG_ERR, "out of memory allocating a Map" );
	    return (void*) 0;
	    }
	++alloc_count;
	}

    /* Fill in the Map entry. */
    m->ino = sb.st_ino;
    m->dev = sb.st_dev;
    m->size = sb.st_size;
    m->ctime = sb.st_ctime;
    m->refcount = 1;
    m->reftime = now;

#ifdef USE_KQUEUE_VNODE
    if (NULL == (m->filename = strdup(filename))) {
      fprintf(stderr, "Out of memory.\n");
      exit(1);
    }
    if (NULL != fnhashP)
      m->fn_hash = *fnhashP;
    else
      m->fn_hash = mmc_stathash(filename);
      
    m->sb = sb;
    m->sbvalidity = 0; /* Initially set to 0, reconfirm on 1st use */
#endif
    /* No need to initialize lruList since we are clearly not on the list */
    time_rfc1123(m->mtime, sizeof(m->mtime), sb.st_mtime);

    figure_mime(hc);

    if (('\0' == hc->encodings[0]) && (sb.st_size <= (off_t)INT_MAX)) {
      char type[200];
      m->fileType = hc->type;
      snprintf(type, sizeof(type), hc->type, hc->hs->charset);
      m->fileInfoLen = snprintf(m->fileInfo, sizeof(m->fileInfo),
				"Content-Type: %s\r\n"
				"Last-Modified: %s"
				"Content-Length: %d\r\n",
				type, m->mtime, (int)sb.st_size);
      if (m->fileInfoLen > sizeof(m->fileInfo)) { /* slow path */
	*m->fileInfo = '\0';
	m->fileInfoLen = 0;
      }
    }
    else { /* treat this in the slow path to avoid mallocing encodings later */
      m->fileType = "";
      *m->fileInfo = '\0';
      m->fileInfoLen = 0;
    }
    
    /* Avoid doing anything for zero-length files; some systems don't like
    ** to mmap them, other systems dislike mallocing zero bytes.
    */
    if ( m->size == 0 )
	m->addr = (void*) 1;	/* arbitrary non-NULL address */
    else
	{
#ifdef USE_SENDFILE
	m->fd = fd;
#ifdef USE_KQUEUE_VNODE
	fdwatch_add_fd(fd, m, FDW_VNODE);
#endif
#elif defined(HAVE_MMAP)
	/* Map the file into memory. */
	m->addr = mmap( 0, m->size, PROT_READ, MAP_SHARED, fd, 0 );
	if ( m->addr == (void*) -1 )
	    {
	    syslog( LOG_ERR, "mmap - %m" );
	    (void) close( fd );
	    free( (void*) m );
	    --alloc_count;
	    return (void*) 0;
	    }
#else /* HAVE_MMAP */
	/* Read the file into memory. */
	m->addr = (void*) malloc( m->size );
	if ( m->addr == (void*) 0 )
	    {
	    syslog( LOG_ERR, "out of memory storing a file" );
	    (void) close( fd );
	    free( (void*) m );
	    --alloc_count;
	    return (void*) 0;
	    }
	if ( read( fd, m->addr, m->size ) != m->size )
	    {
	    syslog( LOG_ERR, "read - %m" );
	    (void) close( fd );
	    free( (void*) m );
	    --alloc_count;
	    return (void*) 0;
	    }
#endif /* HAVE_MMAP */
	}
#ifndef USE_SENDFILE
    (void) close( fd );
#endif /* !USE_SENDFILE */
    /* Put the Map into the hash table. */
    if ( add_hash( m ) < 0 )
	{
	syslog( LOG_ERR, "add_hash() failure" );
	free( (void*) m );
	--alloc_count;
	return (void*) 0;
	}
    
#ifdef USE_KQUEUE_VNODE
    mmc_statadd(m);
#endif

    /* Put the Map on the active list. */
    m->next = maps;
    maps = m;
    ++map_count;
    *mtime = m->mtime;

    *mapp = m;

#ifdef DEBUG_MMC
    fprintf(stderr, __FUNCTION__ ":%d map_count %d lru_count %d\n",
	    __LINE__, map_count, lru_count);
#endif
    
#ifdef USE_SENDFILE
    return (&m->fd);
#else
    /* And return the address. */
    return m->addr;
#endif
    }


void
mmc_unmap( Map **mapp, void* addr, struct stat* sbP, struct timeval* nowP )
    {
    Map* m = (Map*) 0;
#ifdef DEBUG_MMC
    fprintf(stderr, __FUNCTION__ ":%d map_count %d lru_count %d\n",
	    __LINE__, map_count, lru_count);
#endif
    
    
    /* hykim: mmc opt1 */
    if(*mapp != NULL)
      {
	/*      fprintf(stderr, "mmc_unmap: mapp != NULL\n");*/
        m = *mapp;
      }
    else
      {
    /* Find the Map entry for this address.  First try a hash. */
    if ( sbP != (struct stat*) 0 )
	{
	m = find_hash( sbP->st_ino, sbP->st_dev, sbP->st_size, sbP->st_ctime );
	if ( m != (Map*) 0 && m->addr != addr )
	    m = (Map*) 0;
	}
    /* If that didn't work, try a full search. */
    if ( m == (Map*) 0 )
	for ( m = maps; m != (Map*) 0; m = m->next )
	    if ( m->addr == addr )
		break;
      }

    if ( m == (Map*) 0 )
	syslog( LOG_ERR, "mmc_unmap failed to find entry!" );
    else if ( m->refcount <= 0 )
	syslog( LOG_ERR, "mmc_unmap found zero or negative refcount!" );
    else
	{
	  /* hykim: mmc opt1 */
	  *mapp = NULL;

	--m->refcount;
	if ( nowP != (struct timeval*) 0 )
	    m->reftime = nowP->tv_sec;
	else
	    m->reftime = time( (time_t*) 0 );

	if (0 == m->refcount) {
	  /* insert into LRU list */
	  TAILQ_INSERT_TAIL(&mapLRU, m, lruList);
	  lru_count++;
	}
	}
#ifdef DEBUG_MMC
    fprintf(stderr, __FUNCTION__ ":%d map_count %d lru_count %d\n",
	    __LINE__, map_count, lru_count);
#endif
    
    }


void
mmc_cleanup( struct timeval* nowP )
    {
    time_t now;
    Map** mm;
    Map* m;

    return; /* This function is no longer needed since we now replace */
    
    /* Get the current time, if necessary. */
    if ( nowP != (struct timeval*) 0 )
	now = nowP->tv_sec;
    else
	now = time( (time_t*) 0 );

    /* Really unmap any unreferenced entries older than the age limit. */
    for ( mm = &maps; *mm != (Map*) 0; )
	{
	m = *mm;
	if ( m->refcount == 0 && now - m->reftime >= expire_age ) {
	  TAILQ_REMOVE(&mapLRU, m, lruList);
	  lru_count--;
	    really_unmap( mm );
	}
	else
	    mm = &(*mm)->next;
	}

    /* If there are still too many (or too few) maps, adjust the age limit. */
    if ( map_count > DESIRED_MAX_MAPPED_FILES )
	expire_age = max( ( expire_age * 2 ) / 3, DEFAULT_EXPIRE_AGE / 10 );
    else if ( map_count < DESIRED_MAX_MAPPED_FILES / 2 )
	expire_age = min( ( expire_age * 5 ) / 4, DEFAULT_EXPIRE_AGE * 3 );

    /* Really free excess blocks on the free list. */
    while ( free_count > DESIRED_FREE_COUNT )
	{
	m = free_maps;
	free_maps = m->next;
	--free_count;
	free( (void*) m );
	--alloc_count;
	}
    }


static void
really_unmap( Map** mm )
    {
    Map* m;

    m = *mm;
    if ( m->size != 0 )
	{
#ifdef USE_SENDFILE
#ifdef USE_KQUEUE_VNODE
	mmc_statdel(m);
	if (m->filename != NULL)
	  free( m->filename);
	m->sbvalidity = -1;
	fdwatch_del_fd(m->fd);
#endif
	close(m->fd);
#elif defined(HAVE_MMAP)
	if ( munmap( m->addr, m->size ) < 0 )
	    syslog( LOG_ERR, "munmap - %m" );
#else /* HAVE_MMAP */
	free( (void*) m->addr );
#endif /* HAVE_MMAP */
	}
    /* And move the Map to the free list. */
    *mm = m->next;
    --map_count;
    m->next = free_maps;
    free_maps = m;
    ++free_count;
    /* This will sometimes break hash chains, but that's harmless; the
    ** unmapping code that searches the hash table knows to keep searching.
    */
    hash_table[m->hash_idx] = (Map*) 0;
    }


void
mmc_destroy( void )
    {
    Map* m;

    return; /* this cleanup function is unnecessary */
    
    while ( maps != (Map*) 0 ) {
      if (0 == maps->refcount) {
	TAILQ_REMOVE(&mapLRU, maps, lruList);
	lru_count--;
      }
	really_unmap( &maps );
    }
    while ( free_maps != (Map*) 0 )
	{
	m = free_maps;
	free_maps = m->next;
	--free_count;
	free( (void*) m );
	--alloc_count;
	}
    }


/* Make sure the hash table is big enough. */
static int
check_hash_size( void )
    {
    int i;
    Map* m;

    /* Are we just starting out? */
    if ( hash_table == (Map**) 0 )
	{
	hash_size = INITIAL_HASH_SIZE;
	hash_mask = hash_size - 1;
	}
    /* Is it at least three times bigger than the number of entries? */
    else if ( hash_size >= map_count * 3 )
	return 0;
    else
	{
	/* No, got to expand. */
	free( (void*) hash_table );
	/* Double the hash size until it's big enough. */
	do
	    {
	    hash_size = hash_size << 1;
	    }
	while ( hash_size < map_count * 6 );
	hash_mask = hash_size - 1;
	}
    /* Make the new table. */
    hash_table = (Map**) malloc( hash_size * sizeof(Map*) );
    if ( hash_table == (Map**) 0 )
	return -1;
    /* Clear it. */
    for ( i = 0; i < hash_size; ++i )
	hash_table[i] = (Map*) 0;
    /* And rehash all entries. */
    for ( m = maps; m != (Map*) 0; m = m->next )
	if ( add_hash( m ) < 0 )
	    return -1;
    return 0;
    }


static int
add_hash( Map* m )
    {
    unsigned int h, he, i;

    h = hash( m->ino, m->dev, m->size, m->ctime );
    he = ( h + hash_size - 1 ) & hash_mask;
    for ( i = h; ; i = ( i + 1 ) & hash_mask )
	{
	if ( hash_table[i] == (Map*) 0 )
	    {
	    hash_table[i] = m;
	    m->hash = h;
	    m->hash_idx = i;
	    return 0;
	    }
	if ( i == he )
	    break;
	}
    return -1;
    }


static Map*
find_hash( ino_t ino, dev_t dev, off_t size, time_t ctime )
    {
    unsigned int h, he, i;
    Map* m;

    h = hash( ino, dev, size, ctime );
    he = ( h + hash_size - 1 ) & hash_mask;
    for ( i = h; ; i = ( i + 1 ) & hash_mask )
	{
	m = hash_table[i];
	if ( m == (Map*) 0 )
	    break;
	if ( m->hash == h && m->ino == ino && m->dev == dev &&
	     m->size == size && m->ctime == ctime )
	    return m;
	if ( i == he )
	    break;
	}
    return (Map*) 0;
    }


static unsigned int
hash( ino_t ino, dev_t dev, off_t size, time_t ctime )
    {
    unsigned int h = 177573;

    h ^= ino;
    h += h << 5;
    h ^= dev;
    h += h << 5;
    h ^= size;
    h += h << 5;
    h ^= ctime;

    return h & hash_mask;
    }


/* Generate debugging statistics syslog message. */
void
mmc_logstats( long secs )
    {
    syslog(
	LOG_NOTICE, "  map cache - %d allocated, %d active, %d free; hash size: %d; expire age: %ld",
	alloc_count, map_count, free_count, hash_size, expire_age );
    if ( map_count + free_count != alloc_count )
	syslog( LOG_ERR, "map counts don't add up!" );
    }

#ifdef USE_KQUEUE_VNODE

/* The below code is derived from CRC code that is
 * Copyright (c) 1991, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * James W. Williams of NASA Goddard Space Flight Center.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

static const u_int32_t crctab[] = {
	0x0,
	0x04c11db7, 0x09823b6e, 0x0d4326d9, 0x130476dc, 0x17c56b6b,
	0x1a864db2, 0x1e475005, 0x2608edb8, 0x22c9f00f, 0x2f8ad6d6,
	0x2b4bcb61, 0x350c9b64, 0x31cd86d3, 0x3c8ea00a, 0x384fbdbd,
	0x4c11db70, 0x48d0c6c7, 0x4593e01e, 0x4152fda9, 0x5f15adac,
	0x5bd4b01b, 0x569796c2, 0x52568b75, 0x6a1936c8, 0x6ed82b7f,
	0x639b0da6, 0x675a1011, 0x791d4014, 0x7ddc5da3, 0x709f7b7a,
	0x745e66cd, 0x9823b6e0, 0x9ce2ab57, 0x91a18d8e, 0x95609039,
	0x8b27c03c, 0x8fe6dd8b, 0x82a5fb52, 0x8664e6e5, 0xbe2b5b58,
	0xbaea46ef, 0xb7a96036, 0xb3687d81, 0xad2f2d84, 0xa9ee3033,
	0xa4ad16ea, 0xa06c0b5d, 0xd4326d90, 0xd0f37027, 0xddb056fe,
	0xd9714b49, 0xc7361b4c, 0xc3f706fb, 0xceb42022, 0xca753d95,
	0xf23a8028, 0xf6fb9d9f, 0xfbb8bb46, 0xff79a6f1, 0xe13ef6f4,
	0xe5ffeb43, 0xe8bccd9a, 0xec7dd02d, 0x34867077, 0x30476dc0,
	0x3d044b19, 0x39c556ae, 0x278206ab, 0x23431b1c, 0x2e003dc5,
	0x2ac12072, 0x128e9dcf, 0x164f8078, 0x1b0ca6a1, 0x1fcdbb16,
	0x018aeb13, 0x054bf6a4, 0x0808d07d, 0x0cc9cdca, 0x7897ab07,
	0x7c56b6b0, 0x71159069, 0x75d48dde, 0x6b93dddb, 0x6f52c06c,
	0x6211e6b5, 0x66d0fb02, 0x5e9f46bf, 0x5a5e5b08, 0x571d7dd1,
	0x53dc6066, 0x4d9b3063, 0x495a2dd4, 0x44190b0d, 0x40d816ba,
	0xaca5c697, 0xa864db20, 0xa527fdf9, 0xa1e6e04e, 0xbfa1b04b,
	0xbb60adfc, 0xb6238b25, 0xb2e29692, 0x8aad2b2f, 0x8e6c3698,
	0x832f1041, 0x87ee0df6, 0x99a95df3, 0x9d684044, 0x902b669d,
	0x94ea7b2a, 0xe0b41de7, 0xe4750050, 0xe9362689, 0xedf73b3e,
	0xf3b06b3b, 0xf771768c, 0xfa325055, 0xfef34de2, 0xc6bcf05f,
	0xc27dede8, 0xcf3ecb31, 0xcbffd686, 0xd5b88683, 0xd1799b34,
	0xdc3abded, 0xd8fba05a, 0x690ce0ee, 0x6dcdfd59, 0x608edb80,
	0x644fc637, 0x7a089632, 0x7ec98b85, 0x738aad5c, 0x774bb0eb,
	0x4f040d56, 0x4bc510e1, 0x46863638, 0x42472b8f, 0x5c007b8a,
	0x58c1663d, 0x558240e4, 0x51435d53, 0x251d3b9e, 0x21dc2629,
	0x2c9f00f0, 0x285e1d47, 0x36194d42, 0x32d850f5, 0x3f9b762c,
	0x3b5a6b9b, 0x0315d626, 0x07d4cb91, 0x0a97ed48, 0x0e56f0ff,
	0x1011a0fa, 0x14d0bd4d, 0x19939b94, 0x1d528623, 0xf12f560e,
	0xf5ee4bb9, 0xf8ad6d60, 0xfc6c70d7, 0xe22b20d2, 0xe6ea3d65,
	0xeba91bbc, 0xef68060b, 0xd727bbb6, 0xd3e6a601, 0xdea580d8,
	0xda649d6f, 0xc423cd6a, 0xc0e2d0dd, 0xcda1f604, 0xc960ebb3,
	0xbd3e8d7e, 0xb9ff90c9, 0xb4bcb610, 0xb07daba7, 0xae3afba2,
	0xaafbe615, 0xa7b8c0cc, 0xa379dd7b, 0x9b3660c6, 0x9ff77d71,
	0x92b45ba8, 0x9675461f, 0x8832161a, 0x8cf30bad, 0x81b02d74,
	0x857130c3, 0x5d8a9099, 0x594b8d2e, 0x5408abf7, 0x50c9b640,
	0x4e8ee645, 0x4a4ffbf2, 0x470cdd2b, 0x43cdc09c, 0x7b827d21,
	0x7f436096, 0x7200464f, 0x76c15bf8, 0x68860bfd, 0x6c47164a,
	0x61043093, 0x65c52d24, 0x119b4be9, 0x155a565e, 0x18197087,
	0x1cd86d30, 0x029f3d35, 0x065e2082, 0x0b1d065b, 0x0fdc1bec,
	0x3793a651, 0x3352bbe6, 0x3e119d3f, 0x3ad08088, 0x2497d08d,
	0x2056cd3a, 0x2d15ebe3, 0x29d4f654, 0xc5a92679, 0xc1683bce,
	0xcc2b1d17, 0xc8ea00a0, 0xd6ad50a5, 0xd26c4d12, 0xdf2f6bcb,
	0xdbee767c, 0xe3a1cbc1, 0xe760d676, 0xea23f0af, 0xeee2ed18,
	0xf0a5bd1d, 0xf464a0aa, 0xf9278673, 0xfde69bc4, 0x89b8fd09,
	0x8d79e0be, 0x803ac667, 0x84fbdbd0, 0x9abc8bd5, 0x9e7d9662,
	0x933eb0bb, 0x97ffad0c, 0xafb010b1, 0xab710d06, 0xa6322bdf,
	0xa2f33668, 0xbcb4666d, 0xb8757bda, 0xb5365d03, 0xb1f740b4
};

#define NUMSTATHASHBINS 32768

static TAILQ_HEAD(MapStatHash, MapStruct) statHashBins[NUMSTATHASHBINS];

static
uint32_t mmc_stathash( const char *filename )
{
  uint32_t csum = 0;
  uint32_t len = 0;
  const char *tmpptr;
  
#define	COMPUTE(var, ch)	(var) = (var) << 8 ^ crctab[(var) >> 24 ^ (ch)]

  for (tmpptr = filename; *tmpptr != '\0'; tmpptr++, len++) {
    COMPUTE(csum, *tmpptr);
  }

  /* Include the length of the file. */
  for (; len != 0; len >>= 8) {
    COMPUTE(csum, len & 0xff);
  }

  /* At this point, csum includes the POSIX 1003.2 (CRC) checksum. */
  return(csum);
}

static void
mmc_statadd( Map *map )
{
  uint32_t hashidx;
  
  hashidx = (map->fn_hash) & (NUMSTATHASHBINS-1) ;
  TAILQ_INSERT_HEAD(&statHashBins[hashidx], map, statnext);
}

static void
mmc_statdel( Map *map )
{
  uint32_t hashidx;
  
  hashidx = (map->fn_hash) & (NUMSTATHASHBINS-1) ;
  TAILQ_REMOVE(&statHashBins[hashidx], map, statnext);
}

int
mmc_statlookup( const char *filename, struct stat *sbP, uint32_t *hashP )
{
  uint32_t csum;
  uint32_t hashidx;
  Map *map;

  csum = mmc_stathash(filename);
  if (NULL != hashP)
    *hashP = csum; /* save aside for later use if desired */

  hashidx = csum & (NUMSTATHASHBINS-1) ;
  TAILQ_FOREACH(map, &statHashBins[hashidx], statnext) {
    if ((csum == map->fn_hash) && !strcmp(filename, map->filename)) {
      /* There is a hash entry for this file */
      assert(-1 != map->sbvalidity);
      if (0 == map->sbvalidity) {
	struct stat sb2;
	if ((0 != stat(filename, &sb2)) ||
	    (sb2.st_ino != map->sb.st_ino) ||
	    (sb2.st_dev != map->sb.st_dev) ||
	    (sb2.st_ctime != map->sb.st_ctime)) {
	  Map *mapold = map;
	  /* Something changed about access to file */
	  map->sbvalidity = -1;
	  map = TAILQ_PREV(map, MapStatHash, statnext);
	  TAILQ_REMOVE(&statHashBins[hashidx], mapold, statnext);
	  continue; /* go to next FOREACH entry */
	}
	map->sbvalidity = 1;
      }

      /* Now it's safe to return this */
      *sbP = map->sb;
      return(1);
    }
  }
  return(0);
}

void
mmc_statinval( void *mapPtr )
{
  Map *modFile = (Map *)mapPtr;
  Map *map;
  int hashidx;

#if 0
  fprintf(stderr, "mmc_statinval: file %s\n", modFile->filename);
#endif
  
  if (!fdwatch_check_fd(modFile->fd)) /* something went wrong */
    return;
  
#if 0
  fprintf(stderr, "mmc_statinval after check: file %s\n", modFile->filename);
#endif
  
  modFile->sbvalidity = -1;
  hashidx = modFile->fn_hash & (NUMSTATHASHBINS-1) ;
  TAILQ_FOREACH(map, &statHashBins[hashidx], statnext) {
    if ((modFile->fn_hash == map->fn_hash) &&
	!strcmp(modFile->filename, map->filename)) {
      TAILQ_REMOVE(&statHashBins[hashidx], map, statnext);
      return;
    }
  }

  return;
}

#endif

